
#include "include.h"
#include "cw.h"
#include "io.h"
#include "audio.h"
#include "timer.h"
#include "fifo.h"
#include "cw_chars.h"
#include "dtmf.h"



#define RADIO_NOISE_TIMEOUT   1000            // [ms] Timeout after which the transmission is aborted when channel is used
#define RADIO_TX_DELAY        200             // [ms] Delay between activating TX and start with cw
#define DOT_LENGTH            (CW_DOT_LENGTH) // [ms] Time for one unit = one dot
#define DASH_LENGTH           (3*DOT_LENGTH)  // [ms] Time for one dash
#define INTER_ELEMENT_GAP     (DOT_LENGTH)    // [ms] Time for a gap within a letter
#define SHORT_GAP             (3*DOT_LENGTH)  // [ms] Time for a gap between two letters
#define MEDIUM_GAP            (7*DOT_LENGTH)  // [ms] Time for a gap between two words (Space)
#define FREQUENCY             800             // [Hz] Frequency for the cw tone



FIFO_Init(TextBuffer, char, 30);



uint8_t CW_HandleText(void);
PROGMEM char * CW_GetCode(char character);
PROGMEM char * CW_GetSimpleCode(char character);



void CW_Handle(void)
// ============================================================================
{ // CW_Handle()

  static uint8_t  l_stage;
  static  int8_t  l_dummy;
  static uint16_t l_timer;

  enum
  {
    PTT_IDLE,
    PTT_CHANNELFREE,
    PTT_TX_PRE_DELAY,
    PTT_SEND_DATA,
    PTT_TX_POST_DELAY
  };

  // Handle timer
  l_timer += TMR_GetMs();

  switch(l_stage)
  {
    // Wait for free radio channel
    case PTT_CHANNELFREE:
      if(l_timer >= RADIO_NOISE_TIMEOUT) // Timeout (channel used)?
      {
        while(!FIFO_Empty(TextBuffer))   // Run through buffer
        {
          FIFO_Pop(TextBuffer, l_dummy); // Empty buffer
        }
        l_stage   = PTT_IDLE;            // Restart state machine
        printf("TX aborted:    Channel in use.\n");
      }
      #if(WITH_DTMF)
            if(!DTMF_GetNoise())
      #else
            if(!AUD_NoiseDetected())           // Channel free?
      #endif
      {
        IO_Ptt(TX);                      // Activate transmitter
        l_timer = 0;                     // Restart timer
        l_stage = PTT_TX_PRE_DELAY;      // Goto next stage
      }
      break;

    // Wait for radio to be ready
    case PTT_TX_PRE_DELAY:
      if(l_timer >= RADIO_TX_DELAY)      // Delay reached?
      {
        l_stage = PTT_SEND_DATA;         // Goto next stage
      }
      break;

    // Transmit data
    case PTT_SEND_DATA:
      if(CW_HandleText() == 0)           // Transmission finished?
      {
        l_timer = 0;                     // Restart timer
        l_stage = PTT_TX_POST_DELAY;     // Goto next stage
      }
      break;

    // Give radio some time afterwards
    case PTT_TX_POST_DELAY:
      if(l_timer >= RADIO_TX_DELAY)      // Delay reached?
      {
        IO_Ptt(RX);                      // Activate Receiver
        l_stage = PTT_IDLE;              // Goto next stage
      }
      break;

    // Wait for new data to send
    case PTT_IDLE:
    default:
      if(!FIFO_Empty(TextBuffer))        // New data in Buffer?
      {
        l_timer = 0;                     // Reset timer
        l_stage = PTT_CHANNELFREE;       // Go to next stage
      }
      IO_Ptt(RX);                        // Keep radio in receive mode
      break;
  }

} // CW_Handle()
// ============================================================================



void CW_SetText(char * text)
// ============================================================================
{ // CW_SetText()

  uint8_t l_length;

  for(l_length=0; l_length < strlen(text); l_length++) // Run over given data
  {
    FIFO_Push(TextBuffer, text[l_length]);       // Push data into fifo
  }

  // Debugoutput of data
  printf("TX:            %s\n", text);

} // CW_SetText()
// ============================================================================



uint8_t CW_HandleText(void)
// ============================================================================
{ // CW_HandleText()

  static uint8_t  l_stage;
  static uint8_t  l_char;
  static uint16_t l_timer;
  static uint8_t  l_codepos;
  static char *   l_code;

  enum
  {
    CW_IDLE = 0,
    CW_START,
    CW_HANDLE,
    CW_DOT,
    CW_DASH,
    CW_INTERGAP,
    CW_SHORTGAP,
    CW_MEDIUMGAP
  };

  // Handle timer
  l_timer += TMR_GetMs();

  switch(l_stage)
  {
    // Get new letter
    case CW_START:
      if(!FIFO_Empty(TextBuffer))      // Text in Buffer?
      {
        FIFO_Pop(TextBuffer, l_char);  // Get next char
        if(IO_GetInput(DIP4))          // Simple CW?
        {
          l_code = CW_GetSimpleCode(l_char); // Get simple code
        }
        else                           // Normal CW?
        {
          l_code = CW_GetCode(l_char); // Get code
        }
        l_codepos = 0;                 // Start from first pos
        l_stage = CW_HANDLE;           // Go to next stage
        l_timer=0;                     // Reset timer
      }
      else                             // No text in buffer?
      {
        l_stage = CW_IDLE;             // Reset state machine
      }
      break;

    // Handle current letter
    case CW_HANDLE:
      if(strlen_P(l_code) != 0)        // Symbols in code?
      {
        if(l_codepos < strlen_P(l_code)) // Still parts of the letter do handle?
        {
          if(pgm_read_byte(l_code+l_codepos) == '.') // Dot?
          {
            AUD_SetOutFrequency(FREQUENCY); // Turn on tone
            l_stage = CW_DOT;
          }
          else if(pgm_read_byte(l_code+l_codepos) == '-') // Dash?
          {
            AUD_SetOutFrequency(FREQUENCY); // Turn on tone
            l_stage = CW_DASH;
          }
          else                         // Special gap in Simple CW
          {
            l_stage = CW_INTERGAP;
          }
          l_codepos++;
        }
        else                           // No symbols (-> space)?
        {
          l_stage = CW_SHORTGAP;       // Make a gap
        }
      }
      else                             // Space?
      {
        l_stage = CW_MEDIUMGAP;
      }
      l_timer = 0;                     // Reset timer
      break;

    // Send dot
    case CW_DOT:
      if(l_timer >= DOT_LENGTH)        // Timeout reached?
      {
        AUD_SetOutFrequency(0);        // Turn off tone
        l_stage = CW_INTERGAP;         // Goto next stage
        l_timer = 0;                   // Reset timer
      }
      break;

    // Send dash
    case CW_DASH:
      if(l_timer >= DASH_LENGTH)       // Timeout reached?
      {
        AUD_SetOutFrequency(0);        // Turn off tone
        l_stage = CW_INTERGAP;         // Goto next stage
        l_timer = 0;                   // Reset timer
      }
      break;

    // Inter element gap (within a letter)
    case CW_INTERGAP:
      if(l_timer >= INTER_ELEMENT_GAP) // Timeout reached?
      {
        l_stage = CW_HANDLE;           // Goto next stage
      }
      break;

    // Short gap (between two letters)
    case CW_SHORTGAP:
      if(l_timer >= SHORT_GAP)         // Timeout reached?
      {
        l_stage = CW_START;            // Goto next stage
      }
      break;

    // Medium gap (between two words - space)
    case CW_MEDIUMGAP:
      if(l_timer >= MEDIUM_GAP)        // Timeout reached?
      {
        l_stage = CW_START;            // Goto next stage
      }
      break;

    // Wait for new letters to send
    case CW_IDLE:
    default:
      AUD_SetOutFrequency(0);          // Turn off tone
      l_timer = 0;                     // Reset timer
      if(!FIFO_Empty(TextBuffer))      // Data in Buffer?
      {
        l_stage = CW_START;            // Goto first stage
      }
      break;
  }

  if(    (l_stage != 0)  // State machine still working
      || (!FIFO_Empty(TextBuffer))     // Buffer empty
    )
  {
    return 1;                          // cw output still running
  }
  else
  {
    return 0;                          // cw output finished
  }

} // CW_HandleText()
// ============================================================================



PROGMEM char * CW_GetCode(char character)
// ============================================================================
{ // CW_GetCode()

  switch(tolower(character))
  {
    // --- Letters ---
    case 'a':  return (PROGMEM char *) CW_A;
    case 'b':  return (PROGMEM char *) CW_B;
    case 'c':  return (PROGMEM char *) CW_C;
    case 'd':  return (PROGMEM char *) CW_D;
    case 'e':  return (PROGMEM char *) CW_E;
    case 'f':  return (PROGMEM char *) CW_F;
    case 'g':  return (PROGMEM char *) CW_G;
    case 'h':  return (PROGMEM char *) CW_H;
    case 'i':  return (PROGMEM char *) CW_I;
    case 'j':  return (PROGMEM char *) CW_J;
    case 'k':  return (PROGMEM char *) CW_K;
    case 'l':  return (PROGMEM char *) CW_L;
    case 'm':  return (PROGMEM char *) CW_M;
    case 'n':  return (PROGMEM char *) CW_N;
    case 'o':  return (PROGMEM char *) CW_O;
    case 'p':  return (PROGMEM char *) CW_P;
    case 'q':  return (PROGMEM char *) CW_Q;
    case 'r':  return (PROGMEM char *) CW_R;
    case 's':  return (PROGMEM char *) CW_S;
    case 't':  return (PROGMEM char *) CW_T;
    case 'u':  return (PROGMEM char *) CW_U;
    case 'v':  return (PROGMEM char *) CW_V;
    case 'w':  return (PROGMEM char *) CW_W;
    case 'x':  return (PROGMEM char *) CW_X;
    case 'y':  return (PROGMEM char *) CW_Y;
    case 'z':  return (PROGMEM char *) CW_Z;

    // --- Numbers ---
    case '0':  return (PROGMEM char *) CW_0;
    case '1':  return (PROGMEM char *) CW_1;
    case '2':  return (PROGMEM char *) CW_2;
    case '3':  return (PROGMEM char *) CW_3;
    case '4':  return (PROGMEM char *) CW_4;
    case '5':  return (PROGMEM char *) CW_5;
    case '6':  return (PROGMEM char *) CW_6;
    case '7':  return (PROGMEM char *) CW_7;
    case '8':  return (PROGMEM char *) CW_8;
    case '9':  return (PROGMEM char *) CW_9;

    // --- Punctuation ---
    case '.':  return (PROGMEM char *) CW_PERIOD;
    case ',':  return (PROGMEM char *) CW_COMMA;
    case '?':  return (PROGMEM char *) CW_QUESTIONMARK;
    case '\'': return (PROGMEM char *) CW_APOSTROPHE;
    case '!':  return (PROGMEM char *) CW_EXCLAMATIONMARK;
    case '/':  return (PROGMEM char *) CW_SLASH;
    case '(':  return (PROGMEM char *) CW_PARENTHESISOPEN;
    case ')':  return (PROGMEM char *) CW_PARENTHESISCLOSE;
    case '&':  return (PROGMEM char *) CW_AMPERSAND;
    case ':':  return (PROGMEM char *) CW_COLON;
    case ';':  return (PROGMEM char *) CW_SEMICOLON;
    case '=':  return (PROGMEM char *) CW_DOUBLEDASH;
    case '+':  return (PROGMEM char *) CW_PLUS;
    case '-':  return (PROGMEM char *) CW_MINUS;
    case '_':  return (PROGMEM char *) CW_UNDERSCORE;
    case '\"': return (PROGMEM char *) CW_QUOTATIONMARK;
    case '$':  return (PROGMEM char *) CW_DOLLARSIGN;
    case '@':  return (PROGMEM char *) CW_ATSIGN;

    // --- Unknown ---
    default:   return (PROGMEM char *) CW_SPACE;
  }

} // CW_GetCode()
// ============================================================================



PROGMEM char * CW_GetSimpleCode(char character)
// ============================================================================
{ // CW_GetSimpleCode()

  switch(tolower(character))
  {

    // --- Numbers ---
    case '0':  return (PROGMEM char *) CW_SIMPLE_0;
    case '1':  return (PROGMEM char *) CW_SIMPLE_1;
    case '2':  return (PROGMEM char *) CW_SIMPLE_2;
    case '3':  return (PROGMEM char *) CW_SIMPLE_3;
    case '4':  return (PROGMEM char *) CW_SIMPLE_4;
    case '5':  return (PROGMEM char *) CW_SIMPLE_5;
    case '6':  return (PROGMEM char *) CW_SIMPLE_6;
    case '7':  return (PROGMEM char *) CW_SIMPLE_7;
    case '8':  return (PROGMEM char *) CW_SIMPLE_8;
    case '9':  return (PROGMEM char *) CW_SIMPLE_9;

    // --- Punctuation ---
    case '-':  return (PROGMEM char *) CW_SIMPLE_MINUS;

    // --- Unknown ---
    default:   return (PROGMEM char *) CW_SIMPLE_SPACE;
  }

} // CW_GetSimpleCode()
// ============================================================================



