
// Include Headers
#include "uart.h"
#include "fifo.h"



FIFO_Init(RcvBuffer, uint8_t,   5);
FIFO_Init(TrmBuffer, uint8_t, 200);



#define UART_BAUD_RATE 19200


// Local Function Declarations
int UART_PrintChar(char c, FILE * stream);



void UART_Init(void)
// ==========================================================================
{ // UART_Init()

  // USART Control and Status Register A
  UCSRA = (0 << TXC)   // USART Transmit Complete
        | (0 << U2X)   // Double the USART Transmission Speed
        | (0 << MPCM); // Multi-processor Communication Mode

  // USART Control and Status Register B
  UCSRB = (1 << RXCIE) // RX Complete Interrupt Enable
        | (0 << TXCIE) // TX Complete Interrupt Enable
        | (0 << UDRIE) // USART Data Register Empty Interrupt Enable
        | (1 << RXEN)  // Receiver Enable
        | (1 << TXEN)  // Transmitter Enable
        | (0 << UCSZ2) // Character Size
        | (0 << RXB8)  // 9th data bit
        | (0 << TXB8); // 9th data bit

  // USART Control and Status Register C
  UCSRC = (1 << URSEL)  // Register Select
        | (0 << UMSEL)  // USART Mode Select --> Asynchronous Mode
        | (0 << UPM1)   // Parity Mode       --> No Parity
        | (0 << UPM0)   // Parity Mode
        | (0 << USBS)   // Stop Bit Select   --> 1-bit
        | (1 << UCSZ1)  // Character Size
        | (1 << UCSZ0)  // Character Size
        | (0 << UCPOL); // Clock Polarity

  // USART Baud Rate Registers
  UBRRH = (uint8_t) (((F_CPU)/((UART_BAUD_RATE)*16L)-1) >> 8);
  UBRRL = (uint8_t)  ((F_CPU)/((UART_BAUD_RATE)*16L)-1);

  // Reassign stdout for printf
  static FILE mystdout = FDEV_SETUP_STREAM(UART_PrintChar, NULL, _FDEV_SETUP_WRITE);
  stdout = &mystdout;

} // UART_Init()
// ============================================================================


#define UART_DisableRxInterrupt() UCSRB  &= ~(1<<RXCIE)
#define UART_EnableRxInterrupt()  UCSRB  |=  (1<<RXCIE)
ISR(USART_RXC_vect)
// ============================================================================
{ // UART_RX_vect()

  if (!FIFO_Full(RcvBuffer))
  {
    FIFO_Push(RcvBuffer, UDR);
  }

} // UART_RX_vect()
// ============================================================================



#define UART_DisableUdreInterrupt()  UCSRB  &= ~(1<<UDRIE);
#define UART_EnableUdreInterrupt()   UCSRB  |=  (1<<UDRIE);
ISR(USART_UDRE_vect)
// ============================================================================
{ // UART_UDRE_vect()

  if (!FIFO_Empty(TrmBuffer))
  {
    FIFO_Pop(TrmBuffer, UDR);    // Fetch next byte from TrmBuffer and send it
  }
  else
  {
    UART_DisableUdreInterrupt(); // Disable this interrupt
  }

} // UART_UDRE_vect()
// ============================================================================



int UART_PrintChar(char c, FILE * stream)
// ============================================================================
{ // UART_PrintChar()

  if (c == '\n')
  {
    UART_PrintChar('\r', stream);
  }

  return UART_SendByte(c);

} // UART_PrintChar()
// ============================================================================



uint8_t UART_SendByte(uint8_t c)
// ============================================================================
{ // UART_SendByte()

  uint8_t l_ret = 1;

  UART_DisableUdreInterrupt();

  if (!FIFO_Full(TrmBuffer))
  {
    FIFO_Push(TrmBuffer, c);
    l_ret = 0;
  }

  UART_EnableUdreInterrupt();

  return l_ret;

} // UART_SendByte()
// ============================================================================



uint8_t UART_RcvByte(uint8_t * c)
// ============================================================================
{ // UART_RcvByte()

  UART_DisableRxInterrupt();

  if(!FIFO_Empty(RcvBuffer))
  {
    FIFO_Pop(RcvBuffer, *c);
    UART_EnableRxInterrupt();
    return 0;
  }
  else
  {
    UART_EnableRxInterrupt();
    return 1;
  }

} // UART_RcvByte()
// ============================================================================

