
#include "include.h"
#include "timer.h"
#include "io.h"
#include "audio.h"
#include "dtmf.h"



uint8_t volatile TMR_cycletime_curr = 0;
uint8_t          TMR_cycletime_last = 0;
uint16_t         TMR_counter_1s     = 0;



void TMR_Init(void)
// ============================================================================
{ // TMR_Init()

  // ---------------------------------------------------------
  // Timer 2 for ms counter

  // Timer/Counter Control Register
  TCCR2 = (0 << FOC2)    // Force Output Compare
        | (1 << WGM21)   // Waveform Generation Mode  --> CTC Mode
        | (0 << WGM20)   // Waveform Generation Mode
        | (0 << COM21)   // Compare Match Output Mode --> Normal port operation
        | (0 << COM20)   // Compare Match Output Mode
        | (1 << CS22)    // Clock Select              --> clk/64 (from prescaler)
        | (0 << CS21)    // Clock Select
        | (0 << CS20);   // Clock Select

  // Output Compare Register
  //   8,00MHz =  8000000 / 64 / 1000 = 125,000
  OCR2 = F_CPU / 64 / 1000; // Calculate value for Compare-Register to produce a compare-match every 1ms

  // Timer/Counter Interrupt Mask Register
  TIMSK |= (0 << TOIE2)   // Timer/Counter2 Overflow Interrupt Enable
         | (1 << OCIE2);  // Timer/Counter2 Output Compare Match Interrupt Enable
  // ---------------------------------------------------------

  // ---------------------------------------------------------
  // Timer 1 for Sound generation
  TCCR1A = (0 << COM1A1) // Compare output mode: Normal port operation
         | (0 << COM1A0)
         | (0 << COM1B1) // Compare output mode: Normal port operation
         | (0 << COM1B0)
         | (0 << FOC1A)
         | (0 << FOC1B)
         | (0 << WGM11)  // CTC
         | (0 << WGM10);

  TCCR1B = (0 << ICNC1)
         | (0 << ICES1)
         | (0 << WGM13)
         | (1 << WGM12)
         | (0 << CS12)   // Clock Select --> clk/8
         | (1 << CS11)
         | (0 << CS10);

  TIMSK |= (0 << TICIE1)
         | (0 << OCIE1A)
         | (0 << OCIE1B)
         | (0 << TOIE1);
  // ---------------------------------------------------------

} // TMR_Init()
// ============================================================================



void TMR_SetSoundPeriodUs(uint16_t time)
// ============================================================================
{ // TMR_SetSoundPeriodUs()

  if(time == 0)
  {
    // Deactivate timer interrupt
    cli();
    TIMSK &= ~(1 << OCIE1A);
    sei();
  }
  else
  {
    // Set timer
    cli();
    OCR1A = time >> 1; // half the time to have 2 interrupts during period
    if(TCNT1 > OCR1A)
    {
      TCNT1 = OCR1A-1;
    }
    TIMSK |=  (1 << OCIE1A);
    sei();
  }

} // TMR_SetSoundPeriodUs()
// ============================================================================



ISR(TIMER1_COMPA_vect)
// ============================================================================
{ // TIMER1_COMPA_vect()

  IO_TogSound();

} // TIMER1_COMPA_vect()
// ============================================================================




ISR(TIMER2_COMP_vect)
// ============================================================================
// 1 ms interrupt
// ============================================================================
{ // TIMER2_COMP_vect()

  TMR_cycletime_curr++;

  // Handle Audio
  #if(WITH_DTMF)
    DTMF_Handle1msInt();
  #else
    AUD_Handle1msInt();
  #endif

} // TIMER2_COMP_vect()
// ============================================================================



void TMR_Handle(void)
// ============================================================================
{ // TMR_Handle()

  // Handle second tact
  if(TMR_counter_1s >= 1000)
  {
    TMR_counter_1s -= 1000;
  }

  // Calculate cycletime
  cli();                                   // Disable interrupts temporarily
  TMR_cycletime_last = TMR_cycletime_curr; // Get last cycletime from interrupt
  TMR_cycletime_curr = 0;                  // Clear cycletime for interrupt
  sei();                                   // Enable interrupts again

  // Count second tact
  TMR_counter_1s += TMR_cycletime_last;

} // TMR_Handle()
// ============================================================================



uint8_t TMR_GetMs(void)
// ============================================================================
{ // TMR_GetMs()

  return TMR_cycletime_last;

} // TMR_GetMs()
// ============================================================================



uint8_t TMR_SecondTact(void)
// ============================================================================
{ // TMR_SecondTact()

  if(TMR_counter_1s >= 1000)
  {
    return 1;
  }
  else
  {
    return 0;
  }

} // TMR_SecondTact()
// ============================================================================
