
#include "include.h"
#include "timer.h"
#include "uart.h"
#include "io.h"
#include "audio.h"
#include "adc.h"
#include "cw.h"
#include "dtmf.h"



#define NAME    "TempCW"
#define VERSION 1



void Init()
// ============================================================================
{ // Init()

  int vercnt;

  // Init IO
  IO_Init();

  // Show version via LEDs
  _delay_ms(1000);
  for(vercnt=0; vercnt<VERSION; vercnt++)
  {
    IO_SetLED(LED_TRX, ON);
    IO_SetLED(LED_IN1, ON);
    IO_SetLED(LED_IN2, ON);
    _delay_ms(250);
    IO_SetLED(LED_TRX, OFF);
    IO_SetLED(LED_IN1, OFF);
    IO_SetLED(LED_IN2, OFF);
    _delay_ms(250);
  }
  _delay_ms(1000);

  // Init Timer
  TMR_Init();

  // Init Uart
  UART_Init();
  printf("\n");
  printf("==> Start\n");
  printf("Software:      %s v%02i\n", NAME, VERSION);
  printf("Call:          %s\n",    CALL);
  #if(WITH_DTMF)
    printf("With DTMF:     yes\n");
  #else
    printf("With DTMF:     no\n");
  #endif
  #if(WITH_ALARM)
    printf("With Alarm:    yes\n");
  #else
    printf("With Alarm:    no\n");
  #endif
  printf("Dotlength:     %i ms\n", CW_DOT_LENGTH);

  // Init ADC
  ADC_Init();

} // Init()
// ============================================================================



int main()
// ============================================================================
{ // main()

  uint8_t  min = 255;
  uint8_t  max =   0;
  uint16_t avg =   0;
  uint8_t  command;

  // Init
  Init();

  // Endless loop
  while(1)
  {
    char text[31] = {0}; // Clear in every cycle

    // Handle Timer
    TMR_Handle();

    // Handle IO
    IO_Handle();

    // Handle Audio Detection (Debouncing)
    #if(!WITH_DTMF)
      AUD_Handle();
    #endif

    // Handle CW Output
    CW_Handle();

    // Handle Status LED
    if     (IO_GetPttTx())                     IO_SetLED(LED_TRX, RED);
    #if(WITH_DTMF)
      else if(DTMF_GetDtmfCode() != DTMF_NONE) IO_SetLED(LED_TRX, GREEN);
      else if(DTMF_GetNoise())                 IO_SetLED(LED_TRX, ORANGE);
    #else
      else if(AUD_SignalDetected())            IO_SetLED(LED_TRX, GREEN);
      else if(AUD_NoiseDetected())             IO_SetLED(LED_TRX, ORANGE);
    #endif
    else                                       IO_SetLED(LED_TRX, OFF);

    // Input 1 LED
    if(IO_GetInput(IN1)) IO_SetLED(LED_IN1, ON);
    else                 IO_SetLED(LED_IN1, OFF);

    // Input 2 LED
    if(IO_GetInput(IN2)) IO_SetLED(LED_IN2, ON);
    else                 IO_SetLED(LED_IN2, OFF);

    // Wait for 1750 Hz signal
    #if(WITH_DTMF)
      if(DTMF_GetDtmfCode() == DTMF_STAR)
    #else
      if(AUD_NewSignalDetected())      // 1750 Hz signal detected
    #endif
    {
      // Data for Simple CW
      if(IO_GetInput(DIP4))          // Simple CW?
      {
        if(IO_GetInput(DIP1)) sprintf(text,                "1%i  ", IO_GetInput(IN1)+1);
        if(IO_GetInput(DIP2)) sprintf(&text[strlen(text)], "2%i  ", IO_GetInput(IN2)+1);
        if(IO_GetInput(DIP3)) sprintf(&text[strlen(text)], "%i",    IO_GetTemp());
      }
      // Data for CW
      else                           // Normal CW?
      {
        if(IO_GetInput(DIP1)) sprintf(text,                "I1 %i ", IO_GetInput(IN1));
        if(IO_GetInput(DIP2)) sprintf(&text[strlen(text)], "I2 %i ", IO_GetInput(IN2));
        if(IO_GetInput(DIP3)) sprintf(&text[strlen(text)], "T %i ",  IO_GetTemp());
                              sprintf(&text[strlen(text)], "%s",     CALL);
      }
      CW_SetText(text);
    }

    // Wait for change in input 1 (Alarm)
    #if(WITH_ALARM)
      if(    (IO_GetInputChanged(IN1)) // Input 1 just changed?
          && (IO_GetInput(DIP1))       // Input 1 should be observed?
        )
      {
        // Data for Simple CW
        if(IO_GetInput(DIP4))          // Simple CW
        {
          sprintf(text, "1%i", IO_GetInput(IN1)+1);
        }
        // Data for CW
        else                           // Normal CW
        {
          sprintf(text,                "I1 %i ", IO_GetInput(IN1));
          sprintf(&text[strlen(text)], "%s",     CALL);
        }
        CW_SetText(text);              // Start CW
      }
    #endif

    // Wait for change in input 2 (Alarm)
    #if(WITH_ALARM)
      if(    (IO_GetInputChanged(IN2)) // Input 2 just changed?
          && (IO_GetInput(DIP2))       // Input 2 should be observed?
        )
      {
        // Data for Simple CW
        if(IO_GetInput(DIP4))          // Simple CW?
        {
          sprintf(text, "2%i", IO_GetInput(IN2)+1);
        }
        // Data for CW
        else                           // Normal CW?
        {
          sprintf(text,                "I2 %i ", IO_GetInput(IN2));
          sprintf(&text[strlen(text)], "%s",     CALL);
        }
        CW_SetText(text);              // Start CW
      }
    #endif

    // Debug output via uart
    if(TMR_GetMs() > max) max = TMR_GetMs();
    if(TMR_GetMs() < min) min = TMR_GetMs();
    command = 0;
    UART_RcvByte(&command);
    if(command == 'd')
    //if(TMR_SecondTact())
    {
      printf  ("Temperature:   %i C\n",  IO_GetTemp());
      #if(WITH_DTMF)
        printf("DTMF Code:     %02xi\n", DTMF_GetDtmfCode());
      #else
        printf("Frequency In:  %i Hz\n", AUD_GetInFrequency());
      #endif
      printf  ("Inputs:        %i,%i\n", IO_GetInput(IN1), IO_GetInput(IN2));
      printf  ("Dip:           %i,%i,%i,%i\n", IO_GetInput(DIP1), IO_GetInput(DIP2), IO_GetInput(DIP3), IO_GetInput(DIP4));
      printf  ("Cycletime:     Min %i ms / Max %i ms / Avg %i ms\n", min, max, 1000/avg);
      avg=0;
    }
    avg++;
  }

} // main()
// ============================================================================


