
#include "include.h"
#include "io.h"
#include "adc.h"
#include "timer.h"



// LED Input 1
#define INIT_LED1()   {SET_BIT(DDRB,  1);}
#define LED1_ON()     {SET_BIT(PORTB, 1);}
#define LED1_OFF()    {CLR_BIT(PORTB, 1);}

// LED Input 2
#define INIT_LED2()   {SET_BIT(DDRB,  0);}
#define LED2_ON()     {SET_BIT(PORTB, 0);}
#define LED2_OFF()    {CLR_BIT(PORTB, 0);}

// LED3 LED
#define INIT_LED3()   {SET_BIT(DDRC,  4); SET_BIT(DDRC,  5);}
#define LED3_RED()    {SET_BIT(PORTC, 4); CLR_BIT(PORTC, 5);}
#define LED3_GREEN()  {CLR_BIT(PORTC, 4); SET_BIT(PORTC, 5);}
#define LED3_ORANGE() {SET_BIT(PORTC, 4); SET_BIT(PORTC, 5);}
#define LED3_OFF()    {CLR_BIT(PORTC, 4); CLR_BIT(PORTC, 5);}

// Toggle Sound
#define INIT_SOUND()  {SET_BIT(DDRC,  1);}
#define SOUND()       {TOG_BIT(PORTC, 1);}

// PTT
#define INIT_PTT()    {SET_BIT(DDRC,  2);}
#define PTT_RX()      {CLR_BIT(PORTC, 2);}
#define PTT_TX()      {SET_BIT(PORTC, 2);}
#define PTT_GET_TX()  (GET_BIT(PORTC, 2))


// DIP1
#define INIT_DIP1()   {CLR_BIT(DDRD,  2); SET_BIT(PORTD, 2);}
#define DIP1()        (!GET_BIT(PIND, 2))

// DIP2
#define INIT_DIP2()   { CLR_BIT(DDRD, 3); SET_BIT(PORTD, 3);}
#define DIP2()        (!GET_BIT(PIND, 3))

// DIP3
#define INIT_DIP3()   { CLR_BIT(DDRD, 4); SET_BIT(PORTD, 4);}
#define DIP3()        (!GET_BIT(PIND, 4))

// DIP4
#define INIT_DIP4()   { CLR_BIT(DDRD, 5); SET_BIT(PORTD, 5);}
#define DIP4()        (!GET_BIT(PIND, 5))

// Input 1
#define INIT_IN1()    { CLR_BIT(DDRD, 6); SET_BIT(PORTD, 6);}
#define IN1()         (!GET_BIT(PIND, 6))

// Input 2
#define INIT_IN2()    { CLR_BIT(DDRD, 7); SET_BIT(PORTD, 7);}
#define IN2()         (!GET_BIT(PIND, 7))



#define DEBOUNCE_TIME  100 // [ms] Time for debouncing inputs



uint8_t IO_cnt[IO_COUNT]; // Debouncing counter
uint8_t IO_act;           // Current undebounced value
uint8_t IO_deb;           // Debounced value
uint8_t IO_last;          // Value in last cycle



uint8_t IO_GetHwInput(uint8_t input);



void IO_Init(void)
// ============================================================================
{ // IO_Init()

  // Init the IO pins
  INIT_LED1();
  INIT_LED2();
  INIT_LED3();
  INIT_SOUND();
  INIT_PTT();
  INIT_DIP1();
  INIT_DIP2();
  INIT_DIP3();
  INIT_DIP4();
  INIT_IN1();
  INIT_IN2();

} // IO_Init()
// ============================================================================



void IO_Handle(void)
// ============================================================================
{ // IO_Handle()

  uint8_t cnt;

  // Remember last debounced status for edge detection
  IO_last = IO_deb;

  for(cnt=0; cnt<IO_COUNT; cnt++)
  {
    // Get current status of input
    if(IO_GetHwInput(cnt)) SET_BIT(IO_act, cnt);
    else                   CLR_BIT(IO_act, cnt);

    // Handle counter
    if(GET_BIT(IO_act, cnt))          // Input active?
    {
      if(IO_cnt[cnt] < DEBOUNCE_TIME) // Need to increase counter?
      {
        IO_cnt[cnt] += TMR_GetMs();   // Increase counter
      }
      else                            // Don't need to increase counter?
      {
        IO_cnt[cnt] = DEBOUNCE_TIME;  // Keep counter on safe value
      }
    }
    else                              // Input inactive?
    {
      if(TMR_GetMs() < IO_cnt[cnt])   // Safe to decrease counter?
      {
        IO_cnt[cnt] -= TMR_GetMs();   // Decrease counter
      }
      else                            // Not safe or no need to decrease counter
      {
        IO_cnt[cnt] = 0;              // Keep counter at zero
      }
    }

    // Set debounced value
    if(IO_cnt[cnt] >= DEBOUNCE_TIME)  // Input debounced active?
    {
      SET_BIT(IO_deb, cnt);           // Remember debounced active
    }
    else if(IO_cnt[cnt] == 0)         // Debounced inactive?
    {
      CLR_BIT(IO_deb, cnt);           // Remember debounced inactive
    }
    else                              // Somewhere in between?
    {
      // Do nothing, keep status as it is
    }
  }

} // IO_Handle()
// ============================================================================



uint8_t IO_GetInput(uint8_t input)
// ============================================================================
{ // IO_GetInput()

  return GET_BIT(IO_deb, input);

} // IO_GetInput()
// ============================================================================



uint8_t IO_GetInputChanged(uint8_t input)
// ============================================================================
{ // IO_GetInputChanged()

  if(GET_BIT(IO_deb, input) != GET_BIT(IO_last, input))
  {
    return 1;
  }
  else
  {
    return 0;
  }

} // IO_GetInputChanged()
// ============================================================================



uint8_t IO_GetHwInput(uint8_t input)
// ============================================================================
{ // IO_TogSound()

  switch(input)
  {
    case DIP1: return DIP1();
    case DIP2: return DIP2();
    case DIP3: return DIP3();
    case DIP4: return DIP4();
    case IN1:  return IN1();
    case IN2:  return IN2();
    default:   return 0;
  }

} // IO_TogSound()
// ============================================================================



void IO_TogSound(void)
// ============================================================================
{ // IO_TogSound()

  SOUND();

} // IO_TogSound()
// ============================================================================



void IO_SetLED(uint8_t led, uint8_t status)
// ============================================================================
{ // IO_SetLED()

  if(led == LED_TRX)
  {
    switch(status)
    {
      case ON:
      case RED:
        LED3_RED();
        break;

      case GREEN:
        LED3_GREEN();
        break;

      case ORANGE:
        LED3_ORANGE();
        break;

      default:
        LED3_OFF();
        break;
    }
  }
  else if(led == LED_IN1)
  {
    switch(status)
    {
      case ON:
      case RED:
        LED1_ON();
        break;

      default:
        LED1_OFF();
        break;
    }
  }
  else if(led == LED_IN2)
  {
    switch(status)
    {
      case ON:
      case RED:
        LED2_ON();
        break;

      default:
        LED2_OFF();
        break;
    }
  }

} // IO_SetLED()
// ============================================================================



void IO_Ptt(uint8_t status)
// ============================================================================
{ // IO_Ptt()

  switch(status)
  {
    case TX:
      PTT_TX();
      break;

    default:
      PTT_RX();
      break;
  }

} // IO_Ptt()
// ============================================================================



uint8_t IO_GetPttTx(void)
// ============================================================================
{ // IO_GetPttTx()

  return PTT_GET_TX();

} // IO_GetPttTx()
// ============================================================================



int8_t IO_GetTemp(void)
// ============================================================================
{ // IO_GetTemp()

  uint16_t adc_value;
  int8_t   temperature;

  adc_value = ADC_GetTemp();

  // V = Volt; C = Degree Celsius; K = Degree Kelvin
  // 2,7315 V =    0     C
  // 0,0100 V =    1     K

  // 5,0000 V = 1023     #
  // 0,0049 V =    1     #
  // 2,7315 V =  558,865 # ==> Kelvin_Offset
  // 0,0100 V =    2,046 # ==> Inkrements per Degree


  // Temp = ADC_Value - Kelvin_Offset / Ink_per_Degree
  temperature = ((((int32_t)adc_value * 1000) - 558865) / 2046);

  return temperature;

} // IO_GetTemp()
// ============================================================================



