
#include "include.h"
#include "audio.h"
#include "timer.h"
#include "adc.h"



#define WINDOW          50 // [ms] Frequency detection window
#define SIGNALLENGTH   100 // [ms] Mininum time of signal to detect
#define NOISELENGTH    200 // [ms] Noise detection prolonging time
#define SIGNALFREQ    1750 // [Hz] Signalfrequency
#define SIGNALJITTER    60 // [Hz] Allowed jitter of signalfrequency



uint16_t AUD_counter;
uint16_t AUD_oscillations;
uint16_t AUD_min;
uint16_t AUD_max;
uint16_t AUD_middle;
uint8_t  AUD_noisecnt;
uint8_t  AUD_signalcnt;
uint8_t  AUD_newsignal;



void AUD_SetOutFrequency(uint16_t freq)
// ============================================================================
{ // AUD_SetOutFrequency()

  uint16_t time;

  if(freq > 0)
  {
    time = 1000000 / freq;
  }
  else
  {
    time = 0;
  }

  TMR_SetSoundPeriodUs(time);

} // AUD_SetOutFrequency()
// ============================================================================



#if(!WITH_DTMF)
  void AUD_CntAdcInt(uint16_t adc)
  // ============================================================================
  { // AUD_CntAdcInt()

    static uint8_t last;
    static uint8_t current;

    #define ADC_JITTER 3

    enum
    {
      HIGH,
      LOW
    };

    // Get min and max
    if(adc > AUD_max) AUD_max = adc;
    if(adc < AUD_min) AUD_min = adc;

    // Detect oscillation
    if(adc > (AUD_middle+ADC_JITTER)) current = HIGH;
    if(adc < (AUD_middle-ADC_JITTER)) current = LOW;

    // Count oscillation
    if( (last == HIGH) && (current == LOW) )
    {
      AUD_counter++;
    }
    last = current;

  } // AUD_CntAdcInt()
  // ============================================================================
#endif



#if(!WITH_DTMF)
  uint16_t AUD_GetInFrequency(void)
  // ============================================================================
  { // AUD_GetInFrequency()

    uint16_t l_ret;

    cli();
    l_ret = (1000 / WINDOW) * AUD_oscillations;
    sei();

    return l_ret;

  } // AUD_GetInFrequency()
  // ============================================================================
#endif



#if(!WITH_DTMF)
  void AUD_Handle1msInt(void)
  // ============================================================================
  { // AUD_Handle1msInt()

    static uint8_t  det_timer = 0;

    det_timer++;

    if(det_timer >= WINDOW)
    {
      // Exchange data
      AUD_middle = (AUD_min+AUD_max) / 2; // Write back zeropoint
      AUD_oscillations = AUD_counter;     // Get number of oscillations

      // Prepare next window
      AUD_min     = 0xffff;               // Reset to detect new minimum
      AUD_max     = 0;                    // Reset to detect new maximum
      AUD_counter = 0;                    // Reset counter
      det_timer   = 0;                    // Start new window
    }

  } // AUD_Handle1msInt()
  // ============================================================================
#endif



#if(!WITH_DTMF)
  void AUD_Handle(void)
  // ============================================================================
  { // AUD_Handle()

    uint16_t freq = AUD_GetInFrequency();

    // Detect signal (debounce it)
    AUD_newsignal = 0;
    if(    (freq >= (SIGNALFREQ-SIGNALJITTER)) // Frequency in window?
        && (freq <= (SIGNALFREQ+SIGNALJITTER))
      )
    {
      if(AUD_signalcnt < SIGNALLENGTH)         // Safe to increase mscounter?
      {
        AUD_signalcnt += TMR_GetMs();          // Increase time
        if(AUD_signalcnt >= SIGNALLENGTH)      // Just detected a signal?
        {
          AUD_newsignal = 1;                   // Remember new signal
        }
      }
      else                                     // Time already reached?
      {
        AUD_signalcnt = SIGNALLENGTH;          // Keep time safe
      }
    }
    else                                       // No valid frequency detected?
    {
      AUD_signalcnt = 0;                       // Reset timer
    }

    // Detect noise (prolong it)
    if(freq != 0)                              // Some noise present?
    {
      AUD_noisecnt = NOISELENGTH;              // Set countdown timer to timeout
    }
    else                                       // No noise?
    {
      if(TMR_GetMs() < AUD_noisecnt)           // Is it safe to decrease timer?
      {
        AUD_noisecnt -= TMR_GetMs();           // Decrease timer by ms
      }
      else                                     // Time already passed?
      {
        AUD_noisecnt = 0;                      // Set/keep counter at 0
      }
    }

  } // AUD_Handle()
  // ============================================================================
#endif



#if(!WITH_DTMF)
  uint8_t AUD_SignalDetected(void)
  // ============================================================================
  { // AUD_SignalDetected()

    if(AUD_signalcnt >= SIGNALLENGTH) // Signal seen for wanted time?
    {
      return 1;
    }
    else
    {
      return 0;
    }

  } // AUD_SignalDetected()
  // ============================================================================
#endif



#if(!WITH_DTMF)
  uint8_t AUD_NewSignalDetected(void)
  // ============================================================================
  { // AUD_NewSignalDetected()

    return (AUD_newsignal ? 1 : 0);

  } // AUD_NewSignalDetected()
  // ============================================================================
#endif



#if(!WITH_DTMF)
  uint8_t AUD_NoiseDetected(void)
  // ============================================================================
  { // AUD_NoiseDetected()

    if(AUD_noisecnt != 0)     // Noise present?
    {
      return 1;
    }
    else
    {
      return 0;
    }

  } // AUD_NoiseDetected()
  // ============================================================================
#endif


